using UnityEngine;
using System.Collections;

public class UIInterface : UIInterfaceBase {
	
	[System.Serializable]
	public class FontReference {
		public string fontName;
		
		[System.Serializable]
		public class AlphabetReference {
			public Translator.Alphabet alphabet;
			public UIFont font;
		}
		
		public AlphabetReference[] alphabetReferences;
		
		public UIFont GetFont(Translator.Language language) {
			foreach (AlphabetReference a in alphabetReferences) if (Translator.GetAlphabet(language) == a.alphabet) return a.font;
			return null;
		}
	}
	
	public UIInfo info;
	public UIToggleButton toggleButtonScissors, toggleButtonDrill, toggleButtonInfo;
	public UIRoot uiRoot;
	public Rect[] inputBlocksDefault, inputBlocksInfo;
	
	public float fontSizeMlp = 1;
	public float cyrillicHeadingSizeMlp = 0.8f;
	public FontReference[] fontReferences;
	public UIUsed[] usedButtons;
	
	public bool isUsed {
		get {
			foreach (UIUsed used in usedButtons) {
				if (used.isUsed) return true;
			}
			return false;
		}
	}
	
	public void SetLanguage(Translator translator, Translator.Language language) {
		UILabelMultilanguage[] labels = transform.GetComponentsInChildren<UILabelMultilanguage>();
		
		foreach (UILabelMultilanguage label in labels) {
			label.SetText(translator, language);
			UIFont font = GetFont(label.fontName, language);
			label.SetFont(font);
			float languageMlp = 1;
			if ((label.fontName == "Heading" || label.fontName == "HeadingWeb") && language == Translator.Language.Russian) languageMlp = cyrillicHeadingSizeMlp;
			label.transform.localScale = new Vector3(font.size * fontSizeMlp * languageMlp, font.size * fontSizeMlp * languageMlp, 1);
		}
	}
	
	private UIFont GetFont(string fontName, Translator.Language language) {
		foreach (FontReference f in fontReferences) {
			if (f.fontName == fontName) {
				return f.GetFont(language);
			}
		}
		return null;
	}
	
	public void Activate() {
		gameObject.SetActiveRecursively(true);
	}
	
	public void Deactivate() {
		gameObject.SetActiveRecursively(false);
	}
	
	public bool InputBlocked(Main.InfoState infoState) {
		Rect[] blocks = InputBlocks(infoState);
		int screenH = Screen.height;
		float aspect = (float)screenH / (float)uiRoot.manualHeight;
		float x = Input.mousePosition.x;
		float y = Input.mousePosition.y;
		
		foreach (Rect block in blocks) {
			Rect b = new Rect(block.x * aspect, block.y * aspect, block.width * aspect, block.height * aspect);
			
			if (x > b.x && x < b.x + b.width && screenH - y > b.y && screenH - y < b.y + b.height) return true;
		}
		
		return false;
	}
	
	private Rect[] InputBlocks(Main.InfoState infoState) {
		switch(infoState) {
		case Main.InfoState.Disabled: return inputBlocksDefault;
		default: return inputBlocksInfo;
		}
	}
}
