using UnityEngine;
using System.Collections;

public class UICustomSlider : MonoBehaviour {
	
	public UIPanel panel;
	
	public float min = 0, max = 500;
	public bool invertValue = false;
	public float[] snapValues;
	public float snapRange;
	public float snapSpeed = 5;
	public EasingMode easingMode;
	public float minInertia = 1;
	public float inertiaMlp = 25;
	public float inertiaFalloff = 1;
	public bool mouseScroll = false;
	public float mouseScrollSensitivity = 1;
	[HideInInspector] public bool isUsed = false;
	
	private float lerpFrom = 0;
	private float lerpTo = 0;
	private float lerpTimer = 1;
	public float inertia = 0;
	private bool isHovering = false;
	
	public delegate void OnChanged();
	public OnChanged onChanged;
	public delegate void OnDragged();
	public OnDragged onDragged;
	
	void Awake() {
		onChanged += _OnChanged;
		onDragged += _OnDragged;
	}
	
	private void _OnChanged() {}
	private void _OnDragged() {}
	
	void OnHover(bool isOver) {
		if (!mouseScroll) return;
		isHovering = isOver;
	}
	
	public float value {
		get {
			float v = transform.localPosition.y - min;
			if (invertValue) return 1 - (v / range);
			return v / range;
		}
	}
	
	public void SetValue(float _value) {
		inertia = 0;
		SetPosition(ValueToY(_value));
		onChanged();
	}
	
	public void LerpValue(float _value) {
		inertia = 0;
		lerpFrom = transform.localPosition.y;
		lerpTo = ValueToY(_value);
		lerpTimer = 0;
	}
	
	public void SetRange(float _min, float _max) {
		min = _min;
		max = _max;
		SetPosition(transform.localPosition.y);
		onChanged();
	}
	
	void OnDrag(Vector2 delta) {
		if (!enabled) return;
		if (Input.GetMouseButton(1) || Input.GetMouseButton(2)) return;
		
		SetPosition(transform.localPosition.y + delta.y);

		inertia = delta.y;
		if (Mathf.Abs(inertia) <= minInertia) inertia = 0;
		
		onChanged();
		onDragged();
	}
	
	private void SetPosition(float y) {
		transform.localPosition = new Vector3(transform.localPosition.x, Mathf.Clamp(y, min, max), transform.localPosition.z);
	}
	
	void OnPress(bool pressed) {
		if (!enabled) return;
		isUsed = pressed;
		if (pressed) inertia = 0;
		
		if (!pressed) {
			if (snapValues.Length == 0) return;
			
			float c = closestSnapValue;
			if (Mathf.Abs(value - c) < snapRange) LerpValue(c);
			
		} else lerpTimer = 1;
	}
	
	private float closestSnapValue {
		get {
			float closest = Mathf.Infinity;
			float snapValue = Mathf.Infinity;
			for (int i = 0; i < snapValues.Length; i++) {
				float d = Mathf.Abs(value - snapValues[i]);
				if (d < closest) {
					closest = d;
					snapValue = snapValues[i];
				}
			}
			return snapValue;
		}
	}
	
	private void ApplyInertiaFalloff() {
		if (inertia > 0) inertia = Mathf.Clamp(inertia - inertiaFalloff * Time.deltaTime, 0, Mathf.Infinity);
		else inertia = Mathf.Clamp(inertia + inertiaFalloff * Time.deltaTime, -Mathf.Infinity, 0);
	}
	
	void Update() {
		if (!enabled) return;
		
		if (isHovering && !isUsed) {
			float scroll = Input.GetAxis("Mouse ScrollWheel");
			if (scroll != 0) {
				SetPosition(transform.localPosition.y - mouseScrollSensitivity * scroll);
				onChanged();
			}
		}
		
		if (inertia != 0 && !isUsed) {
			ApplyInertiaFalloff();
			SetPosition(transform.localPosition.y + (inertia * Time.deltaTime * inertiaMlp));
			onChanged();
		}
		
		if (lerpTimer >= 1) return;
		
		lerpTimer = Mathf.Clamp(lerpTimer + Time.deltaTime * snapSpeed, 0, 1);
		float lerpEased = Ease.Float(lerpTimer, easingMode);
			
		SetPosition(Mathf.Lerp(lerpFrom, lerpTo, lerpEased));
		onChanged();
	}
	
	private float ValueToY (float v) {
		if (invertValue) return min + ((1 - v) * range);
		return min + (v * range);
	}
	
	private float range {
		get {
			return max - min;
		}
	}
}
