using UnityEngine;
using System.Collections;

public class CrossSectionTurboSlice : MonoBehaviour {
	
	[System.Serializable]
	public class SliceTarget {
		public Main.LayerName layerName;
		public MeshRenderer renderer;
		public bool recalculateTangents = true;
		
		[HideInInspector] public GameObject slice;
		
		public MeshRenderer activeRenderer {
			get {
				if (slice != null) return slice.renderer as MeshRenderer;
				return renderer;
			}
		}
		
		public void RecalculateTangents() {
			if (!recalculateTangents) return;
			
			MeshFilter meshFilter = slice.GetComponent<MeshFilter>();
			Vector4[] tangents = new Vector4[meshFilter.mesh.vertexCount];
			for (int i = 0; i < tangents.Length; i++) tangents[i] = new Vector4(1, 0, 0, -1);
			meshFilter.mesh.tangents = tangents;
		}
		
		public bool enabled = true;
	}
	
	public TurboSlice turboSlice;
	public SliceTarget[] targets;
	
	#region Public methods
	
	public void SetSliceTarget(Main.LayerName _layerName, bool to) {
		foreach (SliceTarget target in targets) {
			if (target.layerName == _layerName) {
				target.enabled = to;
				target.activeRenderer.enabled = to;
			}
		}
	}
	
	public MeshRenderer GetRenderer(Main.LayerName _layerName) {
		foreach (SliceTarget target in targets) if (target.layerName == _layerName) return target.activeRenderer;
		return null;
	}
	
	public void UpdateSliceTargets() {
		foreach (SliceTarget target in targets) Slice(target);
	}
	
	public void Disable() {
		foreach (SliceTarget target in targets) Reset(target);
	}
	
	#endregion
	
	private Vector3[] tri {
		get {
			return new Vector3[3] {
				transform.position,
				transform.position + transform.up,
				transform.position - transform.forward
			};
		}
	}
	
	private void Slice(SliceTarget sliceTarget) {
		if (sliceTarget.slice != null) GameObject.DestroyImmediate(sliceTarget.slice);
		
		GameObject[] slices = turboSlice.splitByTriangle(sliceTarget.renderer.gameObject, tri, false);
			
		if (slices[0] != sliceTarget.renderer.gameObject) {
			if (slices.Length > 1) GameObject.Destroy(slices[1]);
			sliceTarget.slice = slices[0];
			
			sliceTarget.RecalculateTangents();
		}
		
		float dot = Vector3.Dot(sliceTarget.renderer.bounds.center - transform.position, transform.right);
		bool showMainRenderer = (sliceTarget.slice == null && dot < 0);
		sliceTarget.renderer.enabled = showMainRenderer;
		
		sliceTarget.activeRenderer.enabled = sliceTarget.enabled && (showMainRenderer || sliceTarget.slice != null);
	}
	
	private void Reset(SliceTarget sliceTarget) {
		if (sliceTarget.slice == null) {
			sliceTarget.renderer.enabled = sliceTarget.enabled;
			return;
		}

		GameObject.DestroyImmediate(sliceTarget.slice);
		sliceTarget.renderer.enabled = sliceTarget.enabled;
	}
}
