using UnityEngine;
using System.Collections;

public class CameraViewer : CameraBase {
	
	[System.Serializable]
	public enum Mode {
		None,
		Zoom,
		Pan
	}
	
	public Transform target;
	public float panSpeed = 1, zoomSpeed = 20, minZoom = 9.99f, maxZoom = 100, zOffset = 0.01f;
	
	private Vector3 defaultLocalPosition;
	private Quaternion defaultLocalRotation;
	private Quaternion rotationOffset = Quaternion.identity;
	private Mode mode = Mode.None;
	private bool isMoving = true;
	
	void Start() {
		defaultLocalPosition = target.localPosition;
		defaultLocalRotation = target.localRotation;
		rotationOffset = Quaternion.Euler(0, 270, 0);
	}
	
	public override Vector3 GetPosition() {
		if (inputEnabled) {
			Mode _mode = getMode;
			
			switch(mode) {
			case Mode.None:
				mode = _mode;
				if (mode == Mode.Zoom) StartZoom();
				break;
			case Mode.Zoom:
				if (_mode == Mode.None) mode = Mode.None;
				break;
			default:
				mode = _mode;
				if (mode == Mode.Zoom) StartZoom();
				break;
			}
		} else if (getMode == Mode.None) mode = Mode.None;
		
		Vector3 addVector = Vector3.zero;
		
		switch(mode) {
		case Mode.Zoom:
			addVector = new Vector3(
				zoomAdd * 0.1f * zoomSpeed * target.localPosition.x,
				0,
				0);
			break;
		case Mode.Pan:
			Vector3 panVector = panAdd;
			addVector = new Vector3(0, panVector.y, panVector.x) * panSpeed * target.localPosition.x;
			break;
		}
		
		target.localPosition = new Vector3(
			Mathf.Clamp(target.localPosition.x + addVector.x, minZoom, maxZoom),
			Mathf.Clamp(target.localPosition.y + addVector.y, (-0.5f * target.localPosition.x) - 1, (0.1f * target.localPosition.x) - 1),
			Mathf.Clamp(target.localPosition.z + addVector.z, -70, 70));
		
		clipNear = target.localPosition.x - zOffset;
		
		isMoving = mode != Mode.None;
		
		return target.position;
	}

	public override Quaternion GetRotation() {
		return target.rotation * rotationOffset;
	}
	
	public override void Reset() {
		target.localPosition = defaultLocalPosition;
		target.localRotation = defaultLocalRotation;
	}
	
	public override bool GetMoving() { 
		return isMoving; 
	}
	
	private Mode getMode {
		get {
			if (inputZoom) return Mode.Zoom;
			if (inputPan) return Mode.Pan;
			return Mode.None;
		}
	}
	
	private bool inputZoom {
		get {
			switch(inputMode) {
			case Main.InputMode.MouseKeyboard: return Input.GetAxis("Mouse ScrollWheel") != 0 || (Input.GetMouseButton(0) && (Input.GetMouseButton(1)));
			default: return Input.touchCount >= 2;
			}
		}
	}
	
	private bool inputPan {
		get {
			switch(inputMode) {
			case Main.InputMode.MouseKeyboard: 
				if (Input.GetMouseButton(0) && Input.GetMouseButton(1)) return false;
				return Input.GetMouseButton(0) || Input.GetMouseButton(1) || Input.GetMouseButton(2);
			default: return Input.touchCount == 1;
			}
		}
	}
}
