using UnityEngine;
using System.Collections;

public class UIProfile : MonoBehaviour {
	
	[System.Serializable]
	public class Element {
		public string name;
		public UITexture texture;
		public UILabel label;
	}
	
	public Main main;
	public UIPanel panel;
	public Color colorWater, colorQP;
	public Element[] elements;
	public Drill.Profile profile;
	public EasingMode easingMode;
	public float showSpeed = 4, hideSpeed = 2;
	public float hiddenX = 200;
	
	private Drill.Layer layer;
	private static string suffixShort = "_____ ", suffixLong = "__________ ", suffixUnits = " m";
	private static int showX = -25;

	private float hideTarget = 1, hideT = 1;
	private int size;
	
	void Awake() {
		size = main.references.fontSize;
		UpdateProfile();
		panel.widgetsAreStatic = true;
	}
	
	public void Hide() {
		hideTarget = 1;
	}
	
	public void Show() {
		hideTarget = 0;
		panel.widgetsAreStatic = false;
	}
	
	private int sizeMlp {
		get {
			return size + 1;
		}
	}
	
	private int fontSize {
		get {
			return 22 + (size * 22);
		}
	}
	
	private int layerThicknessX {
		get {
			if (size == 0) return 60;
			return 110;
		}
	}
	
	private int layerDepthX {
		get {
			if (size == 0) return 17;
			return 34;
		}
	}
	
	private int layerDepthLineHeight {
		get {
			if (size == 0) return 65;
			return 75;
		}
	}
	
	private int LayerDepthYOffset {
		get {
			if (size == 0) return 3;
			return 6;
		}
	}
	
	private int layerThicknessLineHeight {
		get {
			if (size == 0) return 20;
			return 40;
		}
	}
	
	private int panelShowX {
		get {
			if (size == 0) return -60;
			return -110;
		}
	}
	
	public void UpdateProfile() {		
		if (profile.layers.Length == 0) {
			foreach (Element element in elements) {
				if (element.texture != null) NGUITools.SetActive(element.texture.gameObject, false);
				if (element.label != null) NGUITools.SetActive(element.label.gameObject, false);
			}
		} else {
			float lastLabelY = Mathf.Infinity;
			
			foreach (Element element in elements) {
				layer = GetLayer(element.name);
				
				if (layer != null) {
					element.texture.transform.localPosition = new Vector3(0, ConvertZToSceen(layer.z), 0);
					element.texture.transform.localScale = new Vector3(20 * sizeMlp, ConvertZToSceen(layer.thickness), 0);
					
					switch(layer.name) {
					case "Aluskord":
						element.label.transform.localPosition = new Vector3(layerDepthX, ConvertZToSceen(layer.z) - LayerDepthYOffset, 0);
						
						if (layer.z > profile.layers[0].z - layerDepthLineHeight) {
							element.label.text = layer.z.ToString() + suffixUnits + suffixLong;
						} else element.label.text = layer.z.ToString() + suffixUnits + suffixShort;
						
						break;
					default:
						lastLabelY = Mathf.Min(ConvertZToSceen(layer.z), lastLabelY - layerThicknessLineHeight);
						element.label.transform.localPosition = new Vector3(layerThicknessX, lastLabelY, 0);
						element.label.text = layer.thickness.ToString() + suffixUnits;
						break;
					}
					
					element.label.transform.localScale = new Vector3(fontSize, fontSize, 1);
					
					NGUITools.SetActive(element.texture.gameObject, true);
					NGUITools.SetActive(element.label.gameObject, true);
				} else {
					switch(element.name) {
					case "Z":
						element.label.transform.localPosition = new Vector3(layerDepthX, ConvertZToSceen(profile.layers[0].z) - LayerDepthYOffset, 0);
						element.label.text = profile.layers[0].z.ToString() + suffixUnits + suffixShort;
						element.label.transform.localScale = new Vector3(fontSize, fontSize, 1);
						
						if (profile.layers[0].name == "Water") element.label.color = colorWater;
						else element.label.color = colorQP;
						
						NGUITools.SetActive(element.label.gameObject, true);
						break;
					default:
						NGUITools.SetActive(element.texture.gameObject, false);
						NGUITools.SetActive(element.label.gameObject, false);
						break;
					}
				}
			}
		}
		
		if (hideT == 0) hideT = 0.01f;
	}
	
	void Update() {
		if (hideT == hideTarget) return;
		if (hideTarget == 0 && profile.layers.Length == 0) return;
		
		hideT = Ease.LerpValue(hideT, hideTarget, 0, 1, hideSpeed, showSpeed);
		
		if (hideT == hideTarget) {	
			if (hideTarget == 1) {
				profile.layers = new Drill.Layer[0];
				panel.widgetsAreStatic = true;
			}
		}
		
		transform.localPosition = Ease.V3(new Vector3(panelShowX + (showX * aspect), -130 * aspect, 0), new Vector3(hiddenX * hideT, -130 * aspect, 0), hideT, easingMode);
	}
	
	private Drill.Layer GetLayer(string layerName) {
		foreach (Drill.Layer l in profile.layers) {
			if (l.name == layerName) return l;
		}
		return null;
	}
	
	private Element GetElement(string layerName) {
		foreach (Element e in elements) {
			if (e.name == layerName) return e;
		}
		return null;
	}
	
	private float ConvertZToSceen(float z) {
		return z * 0.4f * aspect;
	}
	
	private float aspect {
		get {
			return (float)Screen.height / 640.0f;
		}
	}
}
