using UnityEngine;
using System.Collections;

public class Main : MonoBehaviour {
	
	[System.Serializable]
	public enum InfoState {
		Disabled,
		Layers,
		LayerInfo
	}
	
	[System.Serializable]
	public enum ToolState {
		None,
		Scissors,
		Drill,
	}
	
	[System.Serializable]
	public enum LayerName {
		NaN,
		Pinnavesi,
		Pinnakate,
		Devon,
		Silur,
		Ordoviitsium,
		Kambrium,
		Ediacara,
		Proterosoikum
	}
	
	[System.Serializable]
	public enum DeviceName {
		Web,
		Standalone,
		Tablet,
		TabletRetina,
		MobileRetina,
		Mobile
	}
	
	[System.Serializable]
	public enum InputMode {
		MouseKeyboard,
		TouchScreen
	}
	
	[System.Serializable]
	public class DeviceReferences {
		
		[System.Serializable]
		public class Optimizations {
			public bool vignetting = true;
			public bool bumpedShaders = true;
			public bool geoMeshScaling = true;
			public bool highPolyGeoMesh = true;
			public bool realtimeCrossSectioning = true;
		}
		
		public DeviceName deviceName;
		public InputMode inputMode;		
		public UIInterface uiInterface;
		public Vector2 defaultResolution = new Vector2(1024, 768);
		
		public int toolSize {
			get {
				if (Screen.height < largeToolSizeScreenHeight) return 0;
				return 1;
			}
		}
		
		public int fontSize {
			get {
				if (Screen.height < largeFontSizeScreenHeight) return 0;
				return 1;
			}
		}
		
		public int largeToolSizeScreenHeight = 1100;
		public int largeFontSizeScreenHeight = 0;
		
		public Optimizations optimizations;
	}
	
	[HideInInspector] public DeviceName deviceName;
	
	public Translator.Language language;
	public Translator translator;
	
	public DeviceReferences[] deviceReferences;
	
	public InfoState infoState = InfoState.Disabled;
	public ToolState toolState = ToolState.None;
	public UIToolsInterface tools;
	public CameraController cameraController;
	public CrossSection crossSection;
	public GeoMeshManager geoMeshManager;
	
	[HideInInspector] public DeviceReferences references = null;
	
	#region Public methods
	
	public void SetInfoState(InfoState _infoState) {
		if (infoState == _infoState) return;
		infoState = _infoState;
		
		switch(infoState) {
		case InfoState.Disabled:
			references.uiInterface.info.Hide();
			references.uiInterface.toggleButtonInfo.SetActive(false);
			break;
		default:
			references.uiInterface.info.Show();
			references.uiInterface.toggleButtonInfo.SetActive(true);
			break;
		}
	}
	
	public void SetToolState(ToolState _toolState) {
		switch(_toolState) {
		case ToolState.Scissors:
			tools.drill.Hide();
			tools.profile.Hide();
			tools.crossSection.Show();
			tools.scaler.Hide();
			references.uiInterface.toggleButtonScissors.SetActive(true);
			references.uiInterface.toggleButtonDrill.SetActive(false);
			break;
		case ToolState.Drill:
			tools.crossSection.Hide();
			tools.drill.Show();
			tools.profile.Show();
			tools.scaler.Hide();
			references.uiInterface.toggleButtonScissors.SetActive(false);
			references.uiInterface.toggleButtonDrill.SetActive(true);
			break;
		case ToolState.None:
			tools.crossSection.Hide();
			tools.drill.Hide();
			tools.profile.Hide();
			tools.scaler.Hide();
			references.uiInterface.toggleButtonScissors.SetActive(false);
			references.uiInterface.toggleButtonDrill.SetActive(false);
			break;
		}
		
		toolState = _toolState;
	}
	
	public void SetLanguage(Translator.Language _language) {
		language = _language;
		if (!Application.isPlaying) references = _references;
		references.uiInterface.SetLanguage(translator, _language);
	}
	
	#endregion Public methods
	
	private DeviceReferences _references {
		get {
			foreach (DeviceReferences d in deviceReferences) if (d.deviceName == deviceName) return d;
			Debug.LogError("DeviceReferences for " + deviceName + " were not found");
			return null;
		}
	}
	
	void Awake() {
		references = _references;
		SetInfoState(InfoState.Disabled);
		SetToolState(ToolState.None);
		
		tools.SetSize(Mathf.Clamp(references.toolSize, 0, 1));
		
		if (!references.optimizations.vignetting) GameObject.Destroy(cameraController.GetComponent<Vignetting>());
		if (!references.optimizations.bumpedShaders) crossSection.useBumpedShaders = false;
		if (references.optimizations.highPolyGeoMesh) geoMeshManager.SetResolution(GeoMeshManager.MeshContainer.Type.HighPoly);
	}
	
	void Update() {
		cameraController.inputEnabled = !references.uiInterface.InputBlocked(infoState);
		/*
		//@todo Remove these
		if (Input.GetKeyUp(KeyCode.E)) SetLanguage(Translator.Language.Estonian);
		if (Input.GetKeyUp(KeyCode.N)) SetLanguage(Translator.Language.English);
		if (Input.GetKeyUp(KeyCode.R)) SetLanguage(Translator.Language.Russian);
		*/
	}
}
