using UnityEngine;
using System.Collections;

public class CrossSection: MonoBehaviour {
	
	[System.Serializable]
	public enum ShaderMode {
		Enabled,
		Disabled,
		DisabledDouble,
		Active
	}
	
	[System.Serializable]
	public class ShaderGroup {
		public string name;
		public bool enabled;
		public string fallBackName;
		public Shader shaderEnabled, shaderDisabled, shaderDisabledDouble, shaderActive;
		
		public Shader GetShader(CrossSection.ShaderMode shaderMode) {
			switch(shaderMode) {
			case CrossSection.ShaderMode.Enabled: return shaderEnabled;
			case CrossSection.ShaderMode.Disabled: return shaderDisabled;
			case CrossSection.ShaderMode.Active: return shaderActive;
			default: return shaderDisabledDouble;
			}
		}
	}
	
	[System.Serializable]
	public class CSObject {
		public Main.LayerName sliceTargetName;
		public string shaderGroupName;
	}
	
	public CrossSectionTurboSlice csTurboSlice;
	public ShaderMode shaderMode = ShaderMode.Enabled;
	public ShaderGroup[] shaderGroups;
	public CSObject[] csObjects;
	
	public bool useBumpedShaders {
		get {
			return GetShaderGroup("Bumped").enabled;
		}
		set { 
			GetShaderGroup("Bumped").enabled = value;
			SetShaders(shaderMode);
		}
	}
	
	private Matrix4x4 m;
	
	void Start() {
		UpdateClippingPlane();
	}
	
	#region Public methods
	
	public void Activate() {
		SetShaders(ShaderMode.Active);
	}
	
	public void Deactivate() {
		if (shaderMode == ShaderMode.Active) SetShaders(ShaderMode.DisabledDouble);
		else SetShaders(ShaderMode.Disabled);
	}
	
	public void Enable() {
		csTurboSlice.Disable();
		SetShaders(ShaderMode.Enabled);
	}
	
	public void Disable() {
		csTurboSlice.Disable();
		SetShaders(ShaderMode.Disabled);
	}
	
	public void DisableDouble() {
		csTurboSlice.UpdateSliceTargets();
		SetShaders(ShaderMode.DisabledDouble);
	}
	
	public void UpdateClippingPlane() {
		if (shaderMode != ShaderMode.Enabled) return;
		
		Quaternion inverseRotation = Quaternion.Inverse(transform.rotation);
		m = Matrix4x4.TRS (Vector3.zero, inverseRotation, Vector3.one);
		Vector3 d = inverseRotation * transform.position;
		float sectionDepth = d.x;
		
		foreach (CSObject csObject in csObjects) UpdateMaterials(GetRenderer(csObject), sectionDepth, m);
	}
	
	#endregion Public methods
	
	private void UpdateMaterials(MeshRenderer r, float sectionDepth, Matrix4x4 m) {
		if (r == null) return;
		
		foreach (Material material in r.sharedMaterials) {
			material.SetFloat("section_depth", sectionDepth);
			material.SetMatrix("_Rotation", m);
		}
	}

	private void SetShaders(ShaderMode _shaderMode) {
		shaderMode = _shaderMode;
		
		foreach (CSObject csObject in csObjects) {
			Shader shader = GetShader(csObject);
			MeshRenderer csRenderer = GetRenderer(csObject);
			if (csRenderer != null) {
				foreach (Material material in csRenderer.materials) {
					material.shader = shader;
				}
			}
		}
		
		UpdateClippingPlane();
	}
	
	private MeshRenderer GetRenderer(CSObject csObject) {
		return csTurboSlice.GetRenderer(csObject.sliceTargetName);
	}
	
	private Shader GetShader(CSObject csObject) {
		return GetShaderGroup(csObject.shaderGroupName).GetShader(shaderMode);
	}
	
	private ShaderGroup GetShaderGroup(string _name) {
		foreach (ShaderGroup shaderGroup in shaderGroups) {
			if (shaderGroup.name == _name) {
				if (shaderGroup.enabled) return shaderGroup;
				return GetShaderGroup(shaderGroup.fallBackName);
			}
		}
		return null;
	}
}
