using UnityEngine;
using System.Collections;

public class CameraOrbiter: CameraBase {
	
	[System.Serializable]
	public enum Mode {
		None,
		Zoom,
		Pan,
		Rotate,
		PanZoom
	}
	
	public Transform target;
	public float distance = 100, minDistance = 15, maxDistance = 100, xSpeed = 7, ySpeed = 7, yMinLimit = 0, yMaxLimit = 70, zoomSpeed = 0.2f, panSpeed = 0.03f;
	
	private float x, y;
	private Vector3 position = Vector3.zero;
	private Quaternion rotation = Quaternion.identity;
	private Mode mode = Mode.None;
	private bool isMoving = true;
	private bool initiated = false;
	
	#region Public methods
	
	public override Vector3 GetPosition() {
		if (!initiated) {
			Vector3 angles = transform.eulerAngles;
			x = angles.y;
			y = angles.x;
			distance = Vector3.Distance(transform.position, target.position);
			
			initiated = true;
		}
		
		Orbit();

		clipNear = nearClipPlane;
		return position;
	}
	
	public override Quaternion GetRotation() {
		return rotation;
	}
	
	public override void Reset() {}
	
	public override bool GetMoving() { 
		return isMoving; 
	}
	
	#endregion
	
	private float nearClipPlane {
		get {
			float distanceF = Mathf.Clamp(distance - 25, 1, 100);
			
			switch(main.references.inputMode) {
			case Main.InputMode.MouseKeyboard: return Mathf.Clamp(distanceF * distanceF * 0.003f, 1, 100);
			default: return Mathf.Clamp(distanceF * distanceF * 0.005f, 2, 100);
			}
		}
	}
	
	private bool inputZoom {
		get {
			switch(inputMode) {
			case Main.InputMode.MouseKeyboard: return Input.GetAxis("Mouse ScrollWheel") != 0 || (Input.GetMouseButton(0) && Input.GetMouseButton(1) && !Input.GetMouseButton(2));
			default: return false;
			}
		}
	}
	
	private bool inputPan {
		get {
			switch(inputMode) {
			case Main.InputMode.MouseKeyboard: return (Input.GetMouseButton(1) || Input.GetMouseButton(2)) && !Input.GetMouseButton(0);
			default: return false;
			}
		}
	}
	
	private bool inputRotate {
		get {
			switch(inputMode) {
			case Main.InputMode.MouseKeyboard: return Input.GetMouseButton(0) && !Input.GetMouseButton(1) && !Input.GetMouseButton(2);
			default: return Input.touchCount == 1;
			}
		}
	}
	
	private bool inputPanZoom {
		get {
			switch(inputMode) {
			case Main.InputMode.MouseKeyboard: return false;
			default: return Input.touchCount >= 2;
			}
		}
	}
	
	private Mode getMode {
		get {
			if (inputZoom) return Mode.Zoom;
			if (inputRotate) return Mode.Rotate;
			if (inputPan) return Mode.Pan;
			if (inputPanZoom) return Mode.PanZoom;
			return Mode.None;
		}
	}
	
	private void StartMode(Mode _mode) {
		mode = _mode;
		
		if (inputMode == Main.InputMode.MouseKeyboard) return;
		
		switch(mode) {
		case Mode.Zoom:
			StartZoom();
			break;
		case Mode.PanZoom:
			StartZoom();
			break;
		}
	}
	
	private void Orbit() {		
		if (inputEnabled) {
			Mode _mode = getMode;
			
			switch(mode) {
			case Mode.None:
				if (_mode != Mode.None) StartMode(_mode);
				break;
			case Mode.Zoom: 
				if (_mode == Mode.None) mode = Mode.None;
				break;
			case Mode.PanZoom:
				if (_mode == Mode.None) mode = Mode.None;
				if (_mode == Mode.Rotate) mode = Mode.Rotate;
				break;
			default:
				if (_mode == Mode.None) mode = Mode.None;
				if (_mode == Mode.Zoom) StartMode(_mode);
				if (_mode == Mode.PanZoom) StartMode(_mode);
				break;
			}
		} else if (getMode == Mode.None) mode = Mode.None;
			
		switch(mode) {
		case Mode.Zoom:
			distance += zoomAdd * distance * zoomSpeed;
			break;
		case Mode.Pan:
			target.position += transform.rotation * (panAdd * distance * panSpeed);

			Vector3 zeroPoint = new Vector3(40, -1.5f, -30);
			Vector3 dir = Vector3.ClampMagnitude(target.position - zeroPoint, 90);
			target.position = zeroPoint + dir;
			break;
		case Mode.Rotate:
			Vector2 rotateAddVector = rotateAdd;
			x += rotateAddVector.x * xSpeed;
			y -= rotateAddVector.y * ySpeed;
			break;
		case Mode.PanZoom:
			distance += zoomAdd * distance * zoomSpeed;
			target.position += transform.rotation * (zoomPanAdd * distance * panSpeed);

			Vector3 zP = new Vector3(40, -1.5f, -30);
			Vector3 d = Vector3.ClampMagnitude(target.position - zP, 90);
			target.position = zP + d;
			break;
		}
		
		distance = Mathf.Clamp(distance, minDistance, maxDistance);
		y = ClampAngle(y, yMinLimit, yMaxLimit);
		rotation = Quaternion.Euler(y, x, 0);
		position = rotation * new Vector3(0, 0, -distance) + target.position;
		
		isMoving = mode != Mode.None;
	}
	
	private static float ClampAngle(float angle, float min, float max) {
		if (angle < -360) angle += 360;
		if (angle > 360) angle -= 360;
		return Mathf.Clamp (angle, min, max);
	}
}

