using UnityEngine;
using System.Collections;

public class CameraController : MonoBehaviour {
	
	public Main main;
	public EasingMode easingModePosition, easingModeRotation, easingModeFOV, easingModeClip;
	public float modeSwitchSpeed = 2;
	public CameraBase[] cameraModes;
	public bool inputEnabled = true;
	
	private float modeT = 1;
	private CameraBase targetMode = null;
	private Vector3 fromPosition = Vector3.zero;
	private Quaternion fromRotation = Quaternion.identity;
	private float fromFOV = 45;
	private float fromClipNear = 10;
	private bool waitForModeSet = false;
	private int mode = 0;
	
	public delegate void ModeSet(int _mode);
	public ModeSet modeSet;
	
	private void OnModeSet(int _mode) {
		for (int i = 0; i < cameraModes.Length; i++) if (i != _mode) cameraModes[i].Reset();
	}
	
	void Awake() {
		modeSet += OnModeSet;
		
		fromPosition = transform.position;
		fromRotation = transform.rotation;
		fromFOV = camera.fieldOfView;
		fromClipNear = camera.nearClipPlane;
		targetMode = cameraModes[0];
	}
	
	#region Public methods
	
	public bool isMoving {
		get {
			return targetMode.GetMoving();
		}
	}
	
	public void SetMode(int _mode) {
		mode = _mode;
		fromPosition = transform.position;
		fromRotation = transform.rotation;
		fromFOV = camera.fieldOfView;
		fromClipNear = camera.nearClipPlane;
		targetMode = cameraModes[_mode];
		modeT = 0;
		waitForModeSet = true;
	}
	
	#endregion Public methods
	
	void LateUpdate() {
		foreach (CameraBase cameramode in cameraModes) cameramode.inputEnabled = inputEnabled;
		
		if (main.tools.isUsed || main.references.uiInterface.isUsed) return;
		
		modeT = Ease.LerpValue(modeT, 1, 0, 1, modeSwitchSpeed, modeSwitchSpeed);
		if (waitForModeSet && modeT == 1) {
			modeSet(mode);
			waitForModeSet = false;
		}
		
		transform.position = targetPosition;
		transform.rotation = targetRotation;
		camera.fieldOfView = targetFOV;
		camera.nearClipPlane = targetClipNear;
	}
	
	private Vector3 targetPosition {
		get {
			if (modeT <= 0) return fromPosition;
			if (modeT >= 1) return targetMode.GetPosition();
			return Ease.V3(fromPosition, targetMode.GetPosition(), modeT, easingModePosition);
		}
	}
	
	private Quaternion targetRotation {
		get {
			if (modeT <= 0) return fromRotation;
			if (modeT >= 1) return targetMode.GetRotation();
			return Quaternion.Slerp(fromRotation, targetMode.GetRotation(), Ease.Float(modeT, easingModeRotation));
		}
	}
	
	private float targetFOV {
		get {
			if (modeT <= 0) return fromFOV;
			if (modeT >= 1) return targetMode.fieldOfView;
			return Mathf.Lerp(fromFOV, targetMode.fieldOfView, Ease.Float(modeT, easingModeFOV));	
		}
	}
	
	private float targetClipNear {
		get {
			if (modeT <= 0) return fromClipNear;
			if (modeT >= 1) return targetMode.clipNear;
			return Mathf.Lerp(fromClipNear, targetMode.clipNear, Ease.Float(modeT, easingModeClip));	
		}
	}
}
